#target illustrator

function autoLayoutAndSave() {
    if (app.documents.length === 0) {
        alert("No document open.");
        return;
    }

    var sourceDoc = app.activeDocument;
    var allDesigns = [];

    // Collect valid design elements
    for (var i = 0; i < sourceDoc.pageItems.length; i++) {
        var it = sourceDoc.pageItems[i];
        if (it.locked || it.hidden) continue;
        if (it.typename === "GroupItem" && it.parent.typename === "Layer") {
            allDesigns.push(it);
        } else if (
            (it.typename === "CompoundPathItem" || it.typename === "PathItem") &&
            it.parent.typename === "Layer"
        ) {
            allDesigns.push(it);
        }
    }

    var designCount = allDesigns.length;
    if (designCount === 0) {
        alert("No valid designs (group/path/compound path) found.");
        return;
    }

    // === NEW: Create new document with fixed width ===
    var fixedWidth = 4500;
    var padding = 100;

    // Step 1: Duplicate all designs to new doc
    var newDoc = app.documents.add(DocumentColorSpace.RGB, fixedWidth, 1000); // temp height
    var pasted = [];
    for (var i = 0; i < allDesigns.length; i++) {
        pasted.push(allDesigns[i].duplicate(newDoc));
    }

    // Step 2: Group all pasted items
    var layoutGroup = newDoc.groupItems.add();
    for (var i = 0; i < pasted.length; i++) {
        pasted[i].moveToBeginning(layoutGroup);
    }

    // Step 3: Scale group proportionally to fit artboard width (4500 - 200 padding)
    var gb = layoutGroup.visibleBounds;
    var gW = gb[2] - gb[0];
    var gH = gb[1] - gb[3];

    var targetWidth = fixedWidth - padding * 2;
    var scaleFactor = (targetWidth / gW) * 100; // %
    layoutGroup.resize(scaleFactor, scaleFactor);

    // Step 4: Recalculate bounds after scaling
    gb = layoutGroup.visibleBounds;
    gW = gb[2] - gb[0];
    gH = gb[1] - gb[3];

    // Artboard height dynamic: group height + padding*2
    var artW = fixedWidth;
    var artH = gH + padding * 2;

    // Illustrator safe values (round to integer)
    artW = Math.round(artW);
    artH = Math.round(artH);

    newDoc.artboards[0].artboardRect = [0, artH, artW, 0];

    // Step 5: Center align group
    var groupBounds = layoutGroup.visibleBounds;
    var groupWidth = groupBounds[2] - groupBounds[0];
    var groupHeight = groupBounds[1] - groupBounds[3];

    var moveX = (artW - groupWidth) / 2 - groupBounds[0];
    var moveY = (artH - groupHeight) / 2 - groupBounds[3];
    layoutGroup.translate(moveX, moveY);

    // === Save as EPS and JPEG ===
    var prefsFile = new File(Folder.userData + "/saveFolderPath.txt");
    var saveFolder;
    if (prefsFile.exists) {
        prefsFile.open("r");
        saveFolder = new Folder(prefsFile.read());
        prefsFile.close();
    }
    if (!saveFolder || !saveFolder.exists) {
        saveFolder = Folder.selectDialog("Select a folder to save your files \n \n© 2025 MASUD. All rights reserved.");
        if (!saveFolder) return;
        prefsFile.open("w");
        prefsFile.write(saveFolder.fsName);
        prefsFile.close();
    }

    var counterFile = new File(saveFolder + "/counter.txt");
    var counter = 10001234;
    if (counterFile.exists) {
        counterFile.open("r");
        counter = parseInt(counterFile.read()) || counter;
        counterFile.close();
    }

    var epsFile, jpgFile;
    do {
        epsFile = new File(saveFolder + "/" + counter + ".eps");
        jpgFile = new File(saveFolder + "/" + counter + ".jpg");
        counter++;
    } while (epsFile.exists || jpgFile.exists);

    var epsOpts = new EPSSaveOptions();
    epsOpts.compatibility = Compatibility.ILLUSTRATOR10;
    epsOpts.preview = EPSPreview.None;
    epsOpts.embedLinkedFiles = true;
    epsOpts.includeDocumentThumbnails = true;
    newDoc.saveAs(epsFile, epsOpts);

    var jpgOpts = new ExportOptionsJPEG();
    jpgOpts.qualitySetting = 100;
    jpgOpts.artBoardClipping = true;
    newDoc.exportFile(jpgFile, ExportType.JPEG, jpgOpts);

    counterFile.open("w");
    counterFile.write(counter);
    counterFile.close();

    newDoc.close(SaveOptions.DONOTSAVECHANGES);
}

autoLayoutAndSave();
